#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdbool.h>
#include <syslog.h>
#include <stdlib.h>
#include <getopt.h>
#include <inttypes.h>
#include "common.h"
#include "ctl.h"
#include "uim.h"
#include "lite-qmi-uim.h"
#include <QmuxTransport.h>
#include <QmiService.h>
#include <CtlService.h>
#include <QmiSyncObject.h>

#include <locale.h>
#include <sys/time.h>
#include "msgid.h"

#include "dev_util.h"
#include "qmux_util.h"
#include "str_util.h"

extern void uim_indication_handler(uint16_t msgid, uint8_t *msg, uint32_t rlen);

extern void GetSlotsStatus(QmiService* pQmiService);
extern void GetCardStatus(QmiService* pQmiService);
extern void GetIccid(QmiService* pQmiService);
extern void EventRegister(QmiService* pQmiService, uint32_t mask);
extern void SwitchSlot(QmiService* pQmiService, uint8_t logical_slot, uint8_t physical_slot);
extern void VerifyPin(QmiService* pQmiService, pack_uim_VerifyPin_t *pVerifyPin);
extern void SetPinProtection(QmiService* pQmiService, pack_uim_SetPinProtection_t *pRetPinProtectionReq);
extern void UnblockPin(QmiService* pQmiService, pack_uim_UnblockPin_t *pUnblockPinReq);
extern void PowerDown(QmiService* pQmiService);
extern void PowerUp(QmiService* pQmiService);
extern void Reset(QmiService* pQmiService);
extern void RefreshOK(QmiService* pQmiService, pack_uim_SLQSUIMRefreshOK_t *pRefreshOK);
extern void RefreshRegister(QmiService* pQmiService, pack_uim_SLQSUIMRefreshRegister_t *pRefreshRegister);
extern void RefreshComplete(QmiService* pQmiService, pack_uim_SLQSUIMRefreshComplete_t *pRefreshComplete);
extern void RfreshGetLastEvent(QmiService* pQmiService, pack_uim_SLQSUIMRefreshGetLastEvent_t *pRefreshGetLastEventReq);
extern void GetFileAttributes(QmiService* pQmiService, pack_uim_SLQSUIMGetFileAttributes_t *pGetFileAttributesReq);
extern void Depersonalization(QmiService* pQmiService);
extern void Authenticate(QmiService* pQmiService);
extern void GetConfiguration(QmiService* pQmiService, uint32_t mask);
extern void ChangePin(QmiService* pQmiService, pack_uim_ChangePin_t *pChangePin);
extern void ReadTransparent(QmiService* pQmiService);

static bool PrintInvalidArguments();
static void PrintUimEventsRegisterOption();
static void PrintUimSwitchSlotOption();
static void PrintUimVerifyPinOption();
static void PrintUimSetPinProtectionOption();
static void PrintUimUnblockPinOption();
static void PrintUimRefereshOkOption();
static void PrintUimRefereshCompleteOption();
static void PrintUimRefereshRegisterOption();
static void PrintUimRefereshGetLastEventOption();
static void PrintUimConfigurationMaskOption();
static void PrintUimChangePinOption();
static void PrintGetFileAttributesOptions();

static void GetUimSessionInformation(char *pCahr, char *pCharEnd, uim_sessionInformation *pSessionInfo);

static CtlService s_CtlService;
static QmiService s_UimService;
static QmuxTransport s_Transport;

//////////////////////////////////////////////////////////
static void PrintPrompt();

enum AppStateType g_AppState = AppUninitialized;

#define DEVICE_PATH_MAX 256
char g_DevicePath[DEVICE_PATH_MAX]={0};

bool g_PrintVersion = false;
int g_mode = QMUX_INTERFACE_UNKNOWN;

#define APPNAME "lite-qmi-uim"
#define VERSION "1.0.2111.0"

///////////////////////////////////////////////////
static void UimIndicationCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	(void)pIndicationCallbackContext;

	unpack_qmi_t rsp_ctx;
	printf (ANSI_COLOR_YELLOW);
	printf("<< receiving uim indication: %s\n", helper_get_resp_ctx(eUIM, qmiPacket, qmiPacketSize, &rsp_ctx));
	printf("msgid 0x%x, type:%x\n", rsp_ctx.msgid,rsp_ctx.type);
	printf (ANSI_COLOR_RESET);
	uim_indication_handler(rsp_ctx.msgid, qmiPacket,  qmiPacketSize);
}

static void PrintPrompt()
{
    printf("\n     1.  Get Slots Status" 
           "\n     2.  Get Card Status" 
           "\n     3.  Get ICCID"
           "\n     4.  Event Register"
           "\n     5.  Switch Slot"
           "\n     6.  Verify PIN"
           "\n     7.  Set PIN Protection"
           "\n     8.  Unblock PIN"
           "\n     9.  Power Down"
           "\n     10.  Power Up"
           "\n     11.  Reset"
           "\n     12.  Refresh OK"
           "\n     13.  Refresh Register"
           "\n     14.  Refresh Complete"
           "\n     15.  Refresh Get Last Event"
           "\n     16.  Get File Attributes"
           "\n     17.  Depersonalization"
           "\n     18.  Authenticate"
           "\n     19.  Get Configuration"
           "\n     20.  Change PIN"
           "\n     21.  Read Transparent"
           "\n     (q)uit to exit: ");
    fflush(stdout);
}
 
static void ExecuteUimTestCaseSelection(void)
{
	char str_return[MAX_USER_INPUT_SIZE];
	PrintPrompt();
	
	while (1)
	{
		fflush(stdin);
		memset (str_return,0,MAX_USER_INPUT_SIZE);

		fgets(str_return,MAX_USER_INPUT_SIZE,stdin);

		switch(g_AppState)
		{
			case AppCommandExecuted:
				g_AppState = AppRunning;
				PrintPrompt();
			break;
			default:
			if (!strcmp(str_return, "\n"))
			{
				g_AppState = AppRunning;
				PrintPrompt();
			}
			break;
		}

		size_t s = strlen(str_return);
		if (s > 0)
			str_return[s - 1] = 0;
		else
			str_return[0] = 0;

		if (!strcmp(str_return, "q"))
		{
			printf("quitting...\n");
			break;
		} 
        else if (!strcmp(str_return, "d"))
		{
			PrintPrompt();
		}
        else if(g_AppState == AppGetEventRegister)
        {
            uint32_t masks = 0x00;
            if (strlen(str_return) > 0)
            {
                uint64_t mask = 0x00;
                char szMask[20] = {0};
		        char *pPos1 = str_return;
		        char *pPos2 = 0;
		        while (pPos1 < (str_return + strlen(str_return)))
		        {
			        pPos2 = strchr(pPos1, '|');
			        if (pPos2 == NULL)
                    {
                        sscanf( pPos1, "%"SCNx64 , &mask);
                        masks |= mask; 
				        break;
                    }
					StrNCpy(szMask, pPos1, pPos2 - pPos1);
                    sscanf( szMask, "%"SCNx64, &mask);
                    masks |= mask; 
                    pPos1 = pPos2 + 1;
                }                
            }
            fprintf(stdout, "Setting UIM events mask = 0x%04X\n", masks); 

            EventRegister(&s_UimService, masks);
            g_AppState = AppCommandExecuted;            
        }
        else if (g_AppState == AppSwitchSlot)
        {
            char *pChar = str_return;
            if (strlen(str_return) != 3 || *(pChar+1) != ',')
            {
                PrintInvalidArguments();
            }
            else
            {          
                uint8_t  logical_slot = 1;
                uint32_t physical_slot = 1;
                logical_slot = atoi( pChar );
                pChar += 2;
                physical_slot = atoi( pChar );

                SwitchSlot(&s_UimService, logical_slot, physical_slot);
                g_AppState = AppCommandExecuted;
            }
        }
        else if(g_AppState == AppVerifyPin)
        {
            char *pChar = str_return;
            char *pCharSI = strstr(pChar, "SI=");
            char *pCharPIN = strstr(pChar, "PIN=");
            char *pCharEPV = strstr(pChar, "EPV=");
            char *pCharPKI = strstr(pChar, "PKI=");
            char *pCharIT = strstr(pChar, "IT=");

            if (!pCharSI || !pCharPIN)
                PrintInvalidArguments();
            else
            {
                pack_uim_VerifyPin_t verifyPin;
                memset (&verifyPin, 0, sizeof(pack_uim_VerifyPin_t));
                char *pPos2 =  NULL;
                GetUimSessionInformation(pCharSI, pCharPIN, &verifyPin.sessionInfo);
                //PIN verify
                while (true)
                {
                    pCharPIN +=4;
                    verifyPin.verifyPIN.pinID = atoi(pCharPIN);
                    ++pCharPIN;
                    if (*pCharPIN != ',')
                    {
                        PrintInvalidArguments();
                        break;
                    }
                    ++pCharPIN;
                    pPos2 =  strchr(pCharPIN, ' ');
                    char szPin[20] = {0};
                    uint32_t idx = 0;
                    if (pPos2)
						StrNCpy(szPin, pCharPIN, pPos2-pCharPIN);
                    else
                        StrCpy(szPin, pCharPIN);
                    
                    verifyPin.verifyPIN.pinLen = 0;
                    for (idx = 0; idx < strlen(szPin); ++idx)
                    {
                        verifyPin.verifyPIN.pinVal[idx] = szPin[idx];
                        ++verifyPin.verifyPIN.pinLen;
                    }
                    
                    verifyPin.pEncryptedPIN1 = NULL;
                    verifyPin.pKeyReferenceID = NULL;
                    verifyPin.pIndicationToken = NULL;

                    if (pCharEPV)
                    {
                        pCharEPV += 4;
                        pPos2 =  strchr(pCharPIN, ' ');
                        uim_encryptedPIN1 encryptedPIN1;
                        encryptedPIN1.pin1Len = 0;

                        if (pPos2)
                            StrNCpy(szPin, pCharEPV, pPos2-pCharEPV);
                        else
                            StrCpy(szPin, pCharEPV);
                        
                        for (idx = 0; idx < strlen(szPin); ++idx)
                        {
                            encryptedPIN1.pin1Val[idx] = szPin[idx];
                            ++encryptedPIN1.pin1Len;
                        }
                        verifyPin.pEncryptedPIN1 = &encryptedPIN1;
                    }

                    uint8_t keyReferenceID = 0;
                    uint32_t indicationToken = 0;
                    if (pCharPKI)
                    {

                        keyReferenceID = atoi(pCharPKI+4);
                        verifyPin.pKeyReferenceID = &keyReferenceID;
                    }

                    if (pCharIT)
                    {
                        indicationToken = atoi(pCharIT+3);
                        verifyPin.pIndicationToken = &indicationToken;
                    } 
                    VerifyPin(&s_UimService, &verifyPin);
                    break;
                }
            }
            g_AppState = AppCommandExecuted;            
        }
        else if (g_AppState == AppSetPinProtection)
        {
            char *pChar = str_return;
            char *pCharSI = strstr(pChar, "SI=");
            char *pCharPIN = strstr(pChar, "PIN=");
            char *pCharPKI = strstr(pChar, "PKI=");
            char *pCharIT = strstr(pChar, "IT=");

            if (!pCharSI || !pCharPIN)
                PrintInvalidArguments();
            else
            {
                pack_uim_SetPinProtection_t setPinProtectionReq;
                memset (&setPinProtectionReq, 0, sizeof(pack_uim_SetPinProtection_t));
                char *pPos2 =  NULL;
                GetUimSessionInformation(pCharSI, pCharPIN, &setPinProtectionReq.sessionInfo);
                //Set PIN protection
                while (true)
                {
                    pCharPIN +=4;
                    setPinProtectionReq.pinProtection.pinID = atoi(pCharPIN);
                    ++pCharPIN;
                    if (*pCharPIN != ',')
                    {
                        PrintInvalidArguments();
                        break;
                    }
                    ++pCharPIN;
                    setPinProtectionReq.pinProtection.pinOperation = atoi(pCharPIN);
                    pCharPIN += 2;

                    pPos2 =  strchr(pCharPIN, ' ');
                    char szPin[20] = {0};
                    uint32_t idx = 0;
                    if (pPos2)
						StrNCpy(szPin, pCharPIN, pPos2-pCharPIN);
                    else
						StrCpy(szPin, pCharPIN);
                    
                    for (idx = 0; idx < strlen(szPin); ++idx)
                    {
                       setPinProtectionReq.pinProtection.pinValue[idx] = szPin[idx];
                       ++setPinProtectionReq.pinProtection.pinLength;
                    }
                    
                    setPinProtectionReq.pKeyReferenceID = NULL;
                    setPinProtectionReq.pIndicationToken = NULL;

                    uint8_t keyReferenceID = 0;
                    uint32_t indicationToken = 0;
                    if (pCharPKI)
                    {

                        keyReferenceID = atoi(pCharPKI+4);
                        setPinProtectionReq.pKeyReferenceID = &keyReferenceID;
                    }

                    if (pCharIT)
                    {
                        indicationToken = atoi(pCharIT+3);
                        setPinProtectionReq.pIndicationToken = &indicationToken;
                    } 
                    SetPinProtection(&s_UimService, &setPinProtectionReq);
                    break;
                }
            }
            g_AppState = AppCommandExecuted;            
        }
        else if(g_AppState == AppUnblockPin)
        {
            char *pChar = str_return;
            char *pCharSI = strstr(pChar, "SI=");
            char *pCharPIN = strstr(pChar, "PIN=");
            char *pCharPKI = strstr(pChar, "PKI=");
            char *pCharIT = strstr(pChar, "IT=");

            if (!pCharSI || !pCharPIN)
                PrintInvalidArguments();
            else
            {
                pack_uim_UnblockPin_t unblockPinReq;
                memset (&unblockPinReq, 0, sizeof(pack_uim_UnblockPin_t));

                char *pPos2 =  NULL;
                GetUimSessionInformation(pCharSI, pCharPIN, &unblockPinReq.sessionInfo);
                //Set PIN protection
                while (true)
                {
                    char szPin[20] = {0};
                    uint32_t idx = 0;

                    pCharPIN +=4;
                    unblockPinReq.pinProtection.pinID = atoi(pCharPIN);
                    ++pCharPIN;
                    if (*pCharPIN != ',')
                    {
                        PrintInvalidArguments();
                        break;
                    }
                    ++pCharPIN;
                    pPos2 =  strchr(pCharPIN, ',');
                    if (!pPos2)
                    {
                        PrintInvalidArguments();
                        break;
                    }                    
					StrNCpy(szPin, pCharPIN, pPos2-pCharPIN);
                    unblockPinReq.pinProtection.pukLen = strlen(szPin);
                    for (idx = 0; idx < unblockPinReq.pinProtection.pukLen; ++idx)
                    {
                        unblockPinReq.pinProtection.pukVal[idx] = szPin[idx];
                    }

                    pCharPIN = pPos2 + 1;

					pPos2 =  strchr(pCharPIN, ' ');
                    if (pPos2)
						StrNCpy(szPin, pCharPIN, pPos2-pCharPIN);
                    else
						StrCpy(szPin, pCharPIN);

                    unblockPinReq.pinProtection.newPINLen = 0;

                    for (idx = 0; idx < strlen(szPin); ++idx)
                    {
                        unblockPinReq.pinProtection.newPINVal[idx] = szPin[idx];
                        ++unblockPinReq.pinProtection.newPINLen;
                    }
                    
                    unblockPinReq.pKeyReferenceID = NULL;
                    unblockPinReq.pIndicationToken = NULL;

                    uint8_t keyReferenceID = 0;
                    uint32_t indicationToken = 0;
                    if (pCharPKI)
                    {

                        keyReferenceID = atoi(pCharPKI+4);
                        unblockPinReq.pKeyReferenceID = &keyReferenceID;
                    }

                    if (pCharIT)
                    {
                        indicationToken = atoi(pCharIT+3);
                        unblockPinReq.pIndicationToken = &indicationToken;
                    } 
                    UnblockPin(&s_UimService, &unblockPinReq);
                    break;
                }
            }
            g_AppState = AppCommandExecuted;            
        }
        else if(g_AppState == AppRefereshOk)
        {
            char *pChar = str_return;
            char *pCharSI = strstr(pChar, "SI=");
            char *pCharOK = strstr(pChar, "OK=");
            if (!pCharSI || !pCharOK)
                PrintInvalidArguments();
            else
            {
                pack_uim_SLQSUIMRefreshOK_t sRefreshOK;
                memset (&sRefreshOK, 0, sizeof(pack_uim_SLQSUIMRefreshOK_t));

                GetUimSessionInformation(pCharSI, pCharOK, &sRefreshOK.sessionInfo);
                sRefreshOK.OKtoRefresh = atoi(pCharOK+3);

                RefreshOK(&s_UimService, &sRefreshOK);
                g_AppState = AppCommandExecuted;
            }
        }
        else if(g_AppState == AppRefereshComplete)
        {
            char *pChar = str_return;
            char *pCharSI = strstr(pChar, "SI=");
            char *pCharCM = strstr(pChar, "CM=");
            if (!pCharSI || !pCharCM)
                PrintInvalidArguments();
            else
            {
                pack_uim_SLQSUIMRefreshComplete_t sRefreshComplete;
                memset (&sRefreshComplete, 0, sizeof(pack_uim_SLQSUIMRefreshComplete_t));
                GetUimSessionInformation(pCharSI, pCharCM, &sRefreshComplete.sessionInfo);
                sRefreshComplete.refreshComplete = atoi(pCharCM+3);

                RefreshComplete(&s_UimService, &sRefreshComplete);
                g_AppState = AppCommandExecuted;
            }
        }
        else if(g_AppState == AppRefereshRegister)
        {
            char *pChar = str_return;
            char *pCharSI = strstr(pChar, "SI=");
            char *pCharRR = strstr(pChar, "RR=");
            if (!pCharSI || !pCharRR)
                PrintInvalidArguments();
            else
            {
                pack_uim_SLQSUIMRefreshRegister_t refreshRegister;
                memset (&refreshRegister, 0, sizeof(pack_uim_SLQSUIMRefreshRegister_t));
                GetUimSessionInformation(pCharSI, pCharRR, &refreshRegister.sessionInfo);

                //Register Refresh
                while (true)
                {
                    pCharRR +=3;
                    refreshRegister.regRefresh.registerFlag = atoi(pCharRR);
                    ++pCharRR;
                    if (*pCharRR != ',')
                    {
                        PrintInvalidArguments();
                        break;
                    }
                    ++pCharRR;
                    refreshRegister.regRefresh.voteForInit = atoi(pCharRR);
                    ++pCharRR;
                    if (*pCharRR != ',')
                    {
                        PrintInvalidArguments();
                        break;
                    }
                    ++pCharRR;
                    refreshRegister.regRefresh.numFiles = 1; //for this test only 1 data set is used
                    refreshRegister.regRefresh.arrfileInfo[0].fileID = atoi(pCharRR);
                    ++pCharRR;
                    if (*pCharRR != ',')
                    {
                        PrintInvalidArguments();
                        break;
                    }
                    ++pCharRR;
                    char szPath[255];
					StrCpy(szPath, pCharRR);

                    refreshRegister.regRefresh.arrfileInfo[0].pathLen = strlen(szPath);
                    //TBD refreshRegister.regRefresh.arrfileInfo[0].path[0] = 1;

                    RefreshRegister(&s_UimService, &refreshRegister);
                    break;
                }
                g_AppState = AppCommandExecuted;
            }
        }
        else if(g_AppState == AppRefereshGetLastEvent)
        {
            char *pChar = str_return;
            char *pCharSI = strstr(pChar, "SI=");
            if (!pCharSI)
                PrintInvalidArguments();
            else
            {
                pack_uim_SLQSUIMRefreshGetLastEvent_t refreshGetLastEvent;
                memset (&refreshGetLastEvent, 0, sizeof(pack_uim_SLQSUIMRefreshGetLastEvent_t));
                char *pCharEnd = pCharSI + strlen(str_return);
                GetUimSessionInformation(pCharSI, pCharEnd, &refreshGetLastEvent.sessionInfo);

                RfreshGetLastEvent(&s_UimService, &refreshGetLastEvent);
            }
            g_AppState = AppCommandExecuted;
        }
        else if(g_AppState == AppGetFileAttributes)
        {
            char *pChar = str_return;
            char *pCharSI = strstr(pChar, "SI=");
            char *pCharFI = strstr(pChar, "FI=");
            char *pCharRI = strstr(pChar, "RI=");
            char *pCharSA = strstr(pChar, "SA=");

            if (!pCharSI || !pCharFI)
                PrintInvalidArguments();
            else
            {
                pack_uim_SLQSUIMGetFileAttributes_t getFileAttributesReq;
                memset (&getFileAttributesReq, 0, sizeof(pack_uim_SLQSUIMGetFileAttributes_t));
                char *pPos2 =  NULL;
                GetUimSessionInformation(pCharSI, pCharFI, &getFileAttributesReq.sessionInfo);

                while (true)
                {
                    //Check user input for FileIndex is not more than 100 characters
                    if (strlen(pCharFI) > 99)
                    {
                        PrintInvalidArguments();
                        break;
                    }

                    char szBuffer[100] = {0};
                    pCharFI +=3;
                    uint32_t fileID = 0;

                    pPos2 = strchr(pCharFI, ',');
                    if (!pPos2 && !pCharRI && !pCharSA)
                    {
                        pPos2 = strchr(pCharFI, ' ');
                    }

                    if (pPos2)
					    StrNCpy(szBuffer, pCharFI, pPos2-pCharFI);
                    else
                        StrCpy(szBuffer, pCharFI);

                    sscanf( pCharFI, "%x" , &fileID);
                    getFileAttributesReq.fileIndex.fileID = (uint16_t)fileID;

                    if (pPos2)
                    {
                        pCharFI = pPos2 + 1;
                        pPos2 =  strchr(pCharFI, ' ');
                        if (pPos2)
                            StrNCpy(szBuffer, pCharFI, pPos2-pCharFI);
                        else
                            StrCpy(szBuffer, pCharFI);
                        
                        char *pFilePath = szBuffer; 
                        int filePathLen = strlen(szBuffer);

                        if (filePathLen > 0)
                        {
                            uint32_t pathValue = 0;
                            char szPath[20] = {0};
                            char *pPos1 = pFilePath;
                            char *pPos2 = 0;
                            int idx = 0;
                            do
                            {
                                memset(szPath, 0, sizeof(szPath));

                                pPos2 = strchr(pPos1, ',');
                                if (pPos2 == NULL)
                                {
                                    sscanf( pPos1, "%x" , &pathValue);
                                    getFileAttributesReq.fileIndex.path[idx] = (uint16_t)pathValue;
                                    ++idx;
                                    break;
                                }
                                StrNCpy(szPath, pPos1, pPos2 - pPos1);
                                sscanf( szPath, "%x", &pathValue);
                                getFileAttributesReq.fileIndex.path[idx] = (uint16_t)pathValue;
                                pPos1 = pPos2 + 1;
                                ++idx;

                            }while (pPos1 < (pFilePath + filePathLen - 1));

                            getFileAttributesReq.fileIndex.pathLen = idx*2;
                        }
                    }

                    uint32_t indication_token = 0;
                    uint8_t uicc_security_attributes = 0;
                    if (pCharRI)
                    {
                        indication_token = atoi(pCharRI+3);
                        getFileAttributesReq.pIndicationToken = &indication_token;
                    }

                    if (pCharSA)
                    {
                        uicc_security_attributes = atoi(pCharSA+3);
                        getFileAttributesReq.pUICCSecurityAttributes = &uicc_security_attributes;
                    } 
                    
                    GetFileAttributes(&s_UimService, &getFileAttributesReq);
                    break;
                }
            }
            g_AppState = AppCommandExecuted;
        }
        else if(g_AppState == AppGetConfigurationMask)
        {
            uint32_t masks = 0x00;
            if (strlen(str_return) > 0)
            {
                uint64_t mask = 0x00;
                char szMask[20] = {0};
		        char *pPos1 = str_return;
		        char *pPos2 = 0;
		        while (pPos1 < (str_return + strlen(str_return)))
		        {
			        pPos2 = strchr(pPos1, '|');
			        if (pPos2 == NULL)
                    {
                        sscanf( pPos1, "%"SCNx64 , &mask);
                        masks |= mask; 
				        break;
                    }
					StrNCpy(szMask, pPos1, pPos2 - pPos1);
                    sscanf( szMask, "%"SCNx64, &mask);
                    masks |= mask; 
                    pPos1 = pPos2 + 1;
                }                
            }
            fprintf(stdout, "Setting UIM configuration mask = 0x%04X\n", masks); 

            GetConfiguration(&s_UimService, masks);
        }
        else if(g_AppState == AppChangePin)
        {
            char *pChar = str_return;
            char *pCharSI = strstr(pChar, "SI=");
            char *pCharPIN = strstr(pChar, "PIN=");
            char *pCharPKI = strstr(pChar, "PKI=");
            char *pCharIT = strstr(pChar, "IT=");

            if (!pCharSI || !pCharPIN)
                PrintInvalidArguments();
            else
            {
                pack_uim_ChangePin_t changePin;
                memset (&changePin, 0, sizeof(pack_uim_ChangePin_t));
                char *pPos2 =  NULL;
                GetUimSessionInformation(pCharSI, pCharPIN, &changePin.sessionInfo);
                //PIN Change Refresh
                while (true)
                {
                    pCharPIN +=4;
                    changePin.changePIN.pinID = atoi(pCharPIN);
                    ++pCharPIN;
                    if (*pCharPIN != ',')
                    {
                        PrintInvalidArguments();
                        break;
                    }
                    ++pCharPIN;
                    pPos2 =  strchr(pCharPIN, ',');
                    if (!pPos2)
                    {
                        PrintInvalidArguments();
                        break;
                    }
                    char szPin[20] = {0};
                    uint32_t idx = 0;
					StrNCpy(szPin, pCharPIN, pPos2-pCharPIN);
                    
                    for (idx = 0; idx < strlen(szPin); ++idx)
                        changePin.changePIN.oldPINVal[idx] = szPin[idx];
                    changePin.changePIN.oldPINLen = idx;
                    pCharPIN = pPos2 + 1;
					if (pCharPKI)
						StrNCpy(szPin, pCharPIN, pCharPKI-pCharPIN);
                    else
						StrCpy(szPin, pCharPIN);
                    
                    changePin.changePIN.pinLen = 0;
                    for (idx = 0; idx < strlen(szPin); ++idx)
                    {
                        changePin.changePIN.pinVal[idx] = szPin[idx];
                        ++changePin.changePIN.pinLen;
                    }
                    
                    changePin.pKeyReferenceID = NULL;
                    changePin.pIndicationToken = NULL;

                    uint8_t keyReferenceID = 0;
                    uint32_t indicationToken = 0;
                    if (pCharPKI)
                    {
                        keyReferenceID = atoi(pCharPKI+4);
                        changePin.pKeyReferenceID = &keyReferenceID;
                    }

                    if (pCharIT)
                    {
                        indicationToken = atoi(pCharIT+3);
                        changePin.pIndicationToken = &indicationToken;
                    } 
                    
                    ChangePin(&s_UimService, &changePin);
                    break;
                }
            }
            g_AppState = AppCommandExecuted;
        }
		else if (g_AppState == AppRunning)
        {
            if (!strcmp(str_return, "1") || !strcmp(str_return, "Get Slots Status"))
            {
                GetSlotsStatus(&s_UimService);
            } 
            else if (!strcmp(str_return, "2") || !strcmp(str_return, "Get Card Status"))
            {
                GetCardStatus(&s_UimService);
            } 
            else if (!strcmp(str_return, "3") || !strcmp(str_return, "Get ICCID"))
            {
                GetIccid(&s_UimService);
            } 
            else if (!strcmp(str_return, "4") || !strcmp(str_return, "Event Register"))
            {                
                PrintUimEventsRegisterOption();
            } 
            else if (!strcmp(str_return, "5") || !strcmp(str_return, "Switch Slot"))
            {
                PrintUimSwitchSlotOption();
            } 
            else if (!strcmp(str_return, "6") || !strcmp(str_return, "Verify PIN"))
            {
                PrintUimVerifyPinOption();                
            } 
            else if (!strcmp(str_return, "7") || !strcmp(str_return, "Set PIN Protection"))
            {
                PrintUimSetPinProtectionOption();
            } 
            else if (!strcmp(str_return, "8") || !strcmp(str_return, "Unblock PIN"))
            {
                PrintUimUnblockPinOption();
            } 
            else if (!strcmp(str_return, "9") || !strcmp(str_return, "Power Down"))
            {
                PowerDown(&s_UimService);
            } 
            else if (!strcmp(str_return, "10") || !strcmp(str_return, "Power Up"))
            {
                PowerUp(&s_UimService);
            } 
            else if (!strcmp(str_return, "11") || !strcmp(str_return, "Reset"))
            {
                Reset(&s_UimService);
            } 
            else if (!strcmp(str_return, "12") || !strcmp(str_return, "Refresh OK"))
            {
                PrintUimRefereshOkOption();
            } 
            else if (!strcmp(str_return, "13") || !strcmp(str_return, "Refresh Register"))
            {
                PrintUimRefereshRegisterOption();
            } 
            else if (!strcmp(str_return, "14") || !strcmp(str_return, "Refresh Complete"))
            {
                PrintUimRefereshCompleteOption();
            } 
            else if (!strcmp(str_return, "15") || !strcmp(str_return, "Refresh Get Last Event"))
            {
                PrintUimRefereshGetLastEventOption();
            } 
            else if (!strcmp(str_return, "16") || !strcmp(str_return, "Get File Attributes"))
            {
                PrintGetFileAttributesOptions();
            } 
            else if (!strcmp(str_return, "17") || !strcmp(str_return, "Depersonalization"))
            {
                Depersonalization(&s_UimService);
            } 
            else if (!strcmp(str_return, "18") || !strcmp(str_return, "Authenticate"))
            {
                Authenticate(&s_UimService);
            } 
            else if (!strcmp(str_return, "19") || !strcmp(str_return, "Get Configuration"))
            {                
                PrintUimConfigurationMaskOption();
            } 
            else if (!strcmp(str_return, "20") || !strcmp(str_return, "Change PIN"))
            {
                PrintUimChangePinOption();
            } 
            else if (!strcmp(str_return, "21") || !strcmp(str_return, "Read Transparent"))
            {
                ReadTransparent(&s_UimService);
            } 
        }	
	}

	return;
}

void PrintUsageManual()
{
    printf( "\r\n" );
    printf( "App usage: \r\n\r\n" );
    printf( "  -d  --device \n" );
    printf( "        absolute path to qmux device\n\n" );
    printf( "  -m  --mbim \n" );
    printf( "        Device is an MBIM interface (defaults to direct QMUX interface)\n\n" );
	printf( "  -q  --qmi\n");
	printf( "        Use direct QMUX interface (QMI over PCIe, or RmNet over USB)\n\n");
	printf( "  -r  --router\n");
	printf( "        Use QMUX Router\n\n");
	printf( "  -h  --help  \n" );
    printf( "        This option prints the usage instructions.\n\n" );
    printf( "  -V  --version  \n" );
    printf( "        This option prints app version.\n\n" );
}

const char * const s_ShortOptions = "d:hmqrV";

const struct option s_LongOptions[] = {
    {"version",  0, NULL, 'V'},
    {"help",   0, NULL, 'h'},
    {"device", 1, NULL, 'd'},
    {"mbim",   0, NULL, 'm'},
    {"qmi",    0, NULL, 'q'},
	{"router", 0, NULL, 'r'},
	{NULL,     0, NULL,  0 }       /* End of list */
};

static bool PrintInvalidArguments()
{
    printf("\n     Invalid arquments."
           "\n     Press<Enter> to go to previous menu.");
    fflush(stdout);
    return true;
}

static void PrintUimEventsRegisterOption()
{
    fprintf( stderr, "\nPlease select UIM Event Register Mask\n"\
                        "  Card status(0x00)\n"\
                        "  SAP connection(0x01)\n"\
                        "  Extended card status(0x04)\n"\
                        "  Close of provisioning sessions(0x08)\n"\
                        "  Physical slot status(0x10)\n"\
                        "  SIM busy status(0x20)\n"\
                        "  Reduced card status(0x40)\n"\
                        "  Recovery complete(0x80)\n"\
                        "  Supply voltage Vcc status(0x0100)\n"\
                        "  Card activation status(0x0200)\n"\
                        "  Remote SimLock configuration(0x0400)\n"\
                        "  SimLock temporary unlock status(0x0800)\n");

    fprintf( stderr, "  ex. 0x01|0x04|0x10\n\n");
    fprintf( stderr, "Events Mask: ");
    g_AppState = AppGetEventRegister;                
}

static void PrintUimSwitchSlotOption()
{
    fprintf( stderr, "\nPlease enter switch slot valures\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  logical slot(Slot 1: 1, Slot 2: 2, Slot 3: 3, Slot 4: 4, Slot 5: 5)\n"\
                        "  physical slot(Slot 1: 1, Slot 2: 2, Slot 3: 3, Slot 4: 4, Slot 5: 5)\n");

    fprintf( stderr, "  ex. 1,3\n");

    g_AppState = AppSwitchSlot;                
}

static void PrintUimVerifyPinOption()
{
    fprintf( stderr, "\nPlease select verify PIN values\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  Session Information(session type, app IDs): SI=type,ids\n"\
                        "  Verify PIN(PIN ID, PIN value): PIN=1|2|3|4,x..x\n"\
                        "  Encrypted Value of PIN1(optional): EPV=x..x\n"\
                        "  Key Ref ID(optional): PKI=1|2|3|4|5|6|7|8\n"\
                        "  Indication Token(optional): IT=x");

    fprintf( stderr, "  ex. SI=1,1,1 PIN=1,1234 EPV=x..x PKI=1 IT=1\n\n");

    g_AppState = AppVerifyPin;                
}

static void PrintUimSetPinProtectionOption()
{
    fprintf( stderr, "\nPlease select Set PIN Protection values\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  Session Information(session type, app IDs): SI=type,ids\n"\
                        "  Set PIN Protection(PIN ID, PIN Operation - Disable/Enable, PIN value): PIN=1|2|3|4,0|1,x..x\n"\
                        "  Key Ref ID(optional): PKI=1|2|3|4|5|6|7|8\n"\
                        "  Indication Token(optional): IT=x");

    fprintf( stderr, "  ex. SI=1,1,1 PIN=1,1,1234 PKI=1 IT=1\n\n");

    g_AppState = AppSetPinProtection;                
}

static void PrintUimUnblockPinOption()
{
    fprintf( stderr, "\nPlease select Unblock PIN values\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  Session Information(session type, app IDs): SI=type,ids\n"\
                        "  Unblock PIN(PIN ID, PIN Unlock key value -ascii, new PIN value): PIN=1|2|3,x..x,y..y\n"\
                        "  Key Ref ID(optional): PKI=1|2|3|4|5|6|7|8\n"\
                        "  Indication Token(optional): IT=x");

    fprintf( stderr, "  ex. SI=1,1,1 PIN=1,a1b2c,1234 PKI=1 IT=1\n\n");

    g_AppState = AppUnblockPin;
}

static void PrintUimRefereshOkOption()
{
    fprintf( stderr, "\nPlease select referesh OK values\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  Session Information (session type, app IDs): SI=type,ids\n"\
                        "  OK for Refresh (0 – Not OK to refresh | 1 – OK to refresh): OK=0|1\n");

    fprintf( stderr, "  ex. SI=1,1,5 OK=1\n\n");

    g_AppState = AppRefereshOk;                
}

static void PrintUimRefereshCompleteOption()
{
    fprintf( stderr, "\nPlease select referesh complete values\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  Session Information (session type, app IDs): SI=type,ids\n"\
                        "  Refresh Complete (0 – Refresh was not completed successfully | 1 – Refresh was completed successfully): CM=0|1\n");

    fprintf( stderr, "  ex. SI=1,1,5 CM=1\n\n");

    g_AppState = AppRefereshComplete;                
}

static void PrintUimRefereshRegisterOption()
{
    fprintf( stderr, "\nPlease select referesh register values\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  Session Information(session type, app IDs): SI=type,ids\n"\
                        "  Rgister Refresh(register flag,vote for init,file id,path value): RR=(0|1),(0|1),2,path\n");

    fprintf( stderr, "  ex. SI=1,1,2 RR=1,1,2,path\n\n");

    g_AppState = AppRefereshRegister;                
}

static void PrintGetFileAttributesOptions()
{
    fprintf( stderr, "\nPlease select Get File Attributes parameters\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  Session Information(session type, app IDs): SI=type,ids\n"\
                        "  File Info(File ID (3GPP 31.102 sec4.7), path): FI=ID,path\n"\
                        "  Response in Indication(optional) (result must be provided in a subsequent indication): RI=(0|1)\n"\
                        "  UICC Security Attributes(optional) (0: Skip security attributes, 1:eturn security attributes): SA=(0|1)\n");

    fprintf( stderr, "  ex1. SI=0,0 FI=0x6F07\n");
    fprintf( stderr, "  ex2. SI=0,0 FI=0x2FE2,3F00\n");
    fprintf( stderr, "  ex3. SI=0,0 FI=0x6FC9,0x3F00,0x7FFF\n\n");

    g_AppState = AppGetFileAttributes;                
}

static void PrintUimRefereshGetLastEventOption()
{
    fprintf( stderr, "\nPlease select referesh get last event values\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  Session Information (session type, app IDs): SI=type,ids\n");

    fprintf( stderr, "  ex. SI=1,1,1\n\n");

    g_AppState = AppRefereshGetLastEvent;
}

static void PrintUimConfigurationMaskOption()
{
    fprintf( stderr, "\nPlease enter Configuration Mask\n"\
                        "  UIM_GET_CONFIGURATION_AUTOMATIC_SELECTION (0x0001) - Automatic selection\n"\
                        "  UIM_GET_CONFIGURATION_PERSONALIZATION_STATUS(0x0002) - Personalization status\n"\
                        "  UIM_GET_CONFIGURATION_HALT_SUBSCRIPTION (0x0004) - Halt subscription\n"\
                        "  UIM_GET_CONFIGURATION_USB_UICC_SUPPORTED (0x0008) - USB UICC is supported\n"\
                        "  UIM_GET_CONFIGURATION_SAP_CLIENT_SUPPORTED (0x0010) - SAP in client mode is supported\n"\
                        "  UIM_GET_CONFIGURATION_PERSO_REACTIVATION_STATUS(0x0020) - Personalization reactivation status\n"\
                        "  UIM_GET_CONFIGURATION_REMOTE_SIMLOCK_STORAGE(0x0040) - Remote SimLock storage\n"\
                        "  UIM_GET_CONFIGURATION_EMERGENCY_ONLY (0x0080) - Emergency Only mode status\n"\
                        "  UIM_GET_CONFIGURATION_EXTENDED_APDU (0x0100) - Extended length APDUs are supported\n"\
                        "  UIM_GET_CONFIGURATION_INACTIVE_SIMLOCK_CONFIG(0x0200) - Inactive SimLock configuration\n");

    fprintf( stderr, "  ex. 0x01|0x04|0x10\n\n");
    fprintf( stderr, "Events Mask: ");
    g_AppState = AppGetConfigurationMask;                
}

static void PrintUimChangePinOption()
{
    fprintf( stderr, "\nPlease select change PIN values\n"\
                        "  Or Press<Enter> to go to previous menu:\n"
                        "  Session Information(session type, app IDs): SI=type,ids\n"\
                        "  Change PIN(PIN ID, old PIN, new PIN): PIN=1|2|3|4,x..x,y..y\n"\
                        "  Key Ref ID(optional): PKI=1|2|3|4|5|6|7|8\n"\
                        "  Indication Token(optional): IT=x");

    fprintf( stderr, "  ex. SI=1,1,1 PIN=1,1234,4321 PKI=1\n\n");

    g_AppState = AppChangePin;                
}

static void GetUimSessionInformation(char *pCharSI, char *pCharEnd, uim_sessionInformation *pSessionInfo)
{
    pSessionInfo->sessionType = 0xFF;
    uint8_t idx_aid = 0;
    char szaid[10] = {0};
    char *pPos1 = pCharSI + 3;
    char *pPos2 = 0;
    while (pPos1 < pCharEnd)
    {
        pPos2 = strchr(pPos1, ',');
        if (pPos2 == NULL)
        {
            pSessionInfo->aid[idx_aid++] = atoi(pPos1);
            break;
        }

		StrNCpy(szaid, pPos1, pPos2 - pPos1);
        if (pSessionInfo->sessionType == 0xFF)
            pSessionInfo->sessionType = atoi(szaid);
        else 
            pSessionInfo->aid[idx_aid++] = atoi(szaid);

        pPos1 = pPos2 + 1;
    }                
    pSessionInfo->aidLength = idx_aid; 
}

static void ParseCommandLine( int argc, char **argv)
{
    int next_option;
    
    /* Parse the command line before doing anything else */
    do
    {
        /* Read the next option until there are no more */
        next_option = getopt_long( argc, argv,
                                   s_ShortOptions,
                                   s_LongOptions, NULL );

        switch( next_option )
        {
            case 'V':
                /* Print usage information */
                g_PrintVersion = true;
                break;
            case 'h':
                /* Print usage information */
                PrintUsageManual();
                exit (0);
                break;
            case 'd':
				StrCpy(g_DevicePath, optarg);
                break;
            case 'm':
                if (g_mode == QMUX_INTERFACE_UNKNOWN)
                    g_mode = QMUX_INTERFACE_MBIM;
                break;
            case 'q':
                if (g_mode == QMUX_INTERFACE_UNKNOWN)
                    g_mode = QMUX_INTERFACE_DIRECT;
                break;
			case 'r':
				if (g_mode == QMUX_INTERFACE_UNKNOWN)
					g_mode = QMUX_INTERFACE_ROUTER;
				break;
			case -1:
                /* Done with options list */
                break;
            default:
                exit(EXIT_FAILURE);
                break;
        }
    }
    while( next_option != -1 );
}

int main(int argc, char **argv)
{
	int ret = FAILURE;
    g_PrintVersion = false;

    ParseCommandLine(argc, argv);
    if (g_PrintVersion)
    {
        printf("\n%s v%s\n\n", APPNAME, VERSION);
       	if (argc == 2)
		    return 0;
    }

    if (OpenTransport(&s_Transport, g_DevicePath, sizeof(g_DevicePath), &g_mode, NULL,
		true, PrintUsageManual, argv[0], false) != 0)
		return 0;

	if (CtlService_Initialize(&s_CtlService, &s_Transport) != SUCCESS)
		printf("CtlService_Initialize failed\n");
	else
	{
		memset(&s_UimService, 0, sizeof(QmiService));

        // We use the Ctl service to initialize the regular services because it knows how to 
        // acquire client IDs for them.

        // Use a do/while for easy exit. We have to clean up.
        do
        {
            // Infrastructure is now ready. Let's create some regular QMI services.

            ret = CtlService_InitializeRegularService(&s_CtlService, &s_UimService, eUIM, UimIndicationCallback, NULL);
            if (ret != SUCCESS)
            {
                printf("InitializeRegularService eUIM failed.\n");
                break;
            }
            g_AppState = AppRunning;
            
            ExecuteUimTestCaseSelection();

        } while (0);		

        // Shut down.
        g_AppState = AppShuttingDown;

        CtlService_ShutDownRegularService(&s_CtlService, &s_UimService);

        CtlService_ShutDown(&s_CtlService);
    }

	QmuxTransport_ShutDown(&s_Transport);
	
	return ret;
}
