/*
 ***************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 *
 * (c) Copyright 2002-2004, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ***************************************************************************

	Module Name:
	dls.c

	Abstract:
	Handle WMM-DLS state machine

	Revision History:
	Who			When			What
	--------	----------		----------------------------------------------
	Rory		2006-2-14		Direct Link Setup
*/
#include "rt_config.h"

/*
    ==========================================================================
    Description:
        dls state machine init, including state transition and timer init
    Parameters:
        Sm - pointer to the dls state machine
    Note:
        The state machine looks like this

                            DLS_IDLE
    MT2_MLME_DLS_REQUEST   MlmeDlsReqAction
    MT2_PEER_DLS_REQUEST   PeerDlsReqAction
    MT2_PEER_DLS_RESPONSE  PeerDlsRspAction
    MT2_MLME_DLS_TEARDOWN  MlmeTearDownAction
    MT2_PEER_DLS_TEARDOWN  PeerTearDownAction

	IRQL = PASSIVE_LEVEL

    ==========================================================================
 */
void DlsStateMachineInit(
    IN PRTMP_ADAPTER pAd,
    IN STATE_MACHINE *Sm,
    OUT STATE_MACHINE_FUNC Trans[])
{
	UCHAR	i;

    StateMachineInit(Sm, Trans, MAX_DLS_STATE, MAX_DLS_MSG, (STATE_MACHINE_FUNC)Drop, DLS_IDLE, DLS_MACHINE_BASE);

    // the first column
    StateMachineSetAction(Sm, DLS_IDLE, MT2_MLME_DLS_REQ, (STATE_MACHINE_FUNC)MlmeDlsReqAction);
    StateMachineSetAction(Sm, DLS_IDLE, MT2_PEER_DLS_REQ, (STATE_MACHINE_FUNC)PeerDlsReqAction);
    StateMachineSetAction(Sm, DLS_IDLE, MT2_PEER_DLS_RSP, (STATE_MACHINE_FUNC)PeerDlsRspAction);
    StateMachineSetAction(Sm, DLS_IDLE, MT2_MLME_DLS_TEAR_DOWN, (STATE_MACHINE_FUNC)MlmeDlsTearDownAction);
    StateMachineSetAction(Sm, DLS_IDLE, MT2_PEER_DLS_TEAR_DOWN, (STATE_MACHINE_FUNC)PeerDlsTearDownAction);
	StateMachineSetAction(Sm, DLS_IDLE, MT2_PEER_DLS_STAKEY, (STATE_MACHINE_FUNC)PeerDlsStaKeyAction);

	for (i=0; i<MAX_NUM_OF_DLS_ENTRY; i++)
	{
		pAd->StaCfg.DLSEntry[i].pAd = pAd;
		RTMPInitTimer(pAd, &pAd->StaCfg.DLSEntry[i].Timer, GET_TIMER_FUNCTION(DlsTimeout), &pAd->StaCfg.DLSEntry[i], FALSE);
	}
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
VOID MlmeDlsReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
	PUCHAR			pOutBuffer = NULL;
	NDIS_STATUS		NStatus;
	ULONG			FrameLen = 0;
	HEADER_802_11	DlsReqHdr;
	PRT_802_11_DLS	pDLS = NULL;
	UCHAR			Category = CATEGORY_DLS;
	UCHAR			Action = ACTION_DLS_REQUEST;
	ULONG			tmp;
	USHORT			reason;
	ULONG			Timeout;
	BOOLEAN			TimerCancelled;

	if(!MlmeDlsReqSanity(pAd, Elem->Msg, Elem->MsgLen, &pDLS, &reason))
		return;

	DBGPRINT(RT_DEBUG_TRACE,("DLS - MlmeDlsReqAction() \n"));

	NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);  //Get an unused nonpaged memory
	if (NStatus != NDIS_STATUS_SUCCESS)
	{
		DBGPRINT(RT_DEBUG_ERROR,("DLS - MlmeDlsReqAction() allocate memory failed \n"));
		return;
	}

	MgtMacHeaderInit(pAd, &DlsReqHdr, SUBTYPE_ACTION, 0, pAd->CommonCfg.Bssid, pAd->CommonCfg.Bssid);

	// Build basic frame first
	MakeOutgoingFrame(pOutBuffer,				&FrameLen,
					sizeof(HEADER_802_11),		&DlsReqHdr,
					1,							&Category,
					1,							&Action,
					6,							&pDLS->MacAddr,
					6,							pAd->CurrentAddress,
					2,							&pAd->StaActive.CapabilityInfo,
					2,							&pDLS->TimeOut,
					1,							&SupRateIe,
					1,							&pAd->MlmeAux.SupRateLen,
					pAd->MlmeAux.SupRateLen,	pAd->MlmeAux.SupRate,
					END_OF_ARGS);

	if (pAd->MlmeAux.ExtRateLen != 0)
	{
		MakeOutgoingFrame(pOutBuffer + FrameLen,	&tmp,
						  1,						&ExtRateIe,
						  1,						&pAd->MlmeAux.ExtRateLen,
						  pAd->MlmeAux.ExtRateLen,	pAd->MlmeAux.ExtRate,
						  END_OF_ARGS);
		FrameLen += tmp;
	}

	RTMPCancelTimer(&pDLS->Timer, &TimerCancelled);
	Timeout = DLS_TIMEOUT;
	RTMPSetTimer(&pDLS->Timer, Timeout);

	MiniportMMRequest(pAd, pOutBuffer, FrameLen);
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
VOID PeerDlsReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
	PUCHAR			pOutBuffer = NULL;
	NDIS_STATUS		NStatus;
	ULONG			FrameLen = 0;
	USHORT			StatusCode = MLME_SUCCESS;
	HEADER_802_11	DlsRspHdr;
	UCHAR			Category = CATEGORY_DLS;
	UCHAR			Action = ACTION_DLS_RESPONSE;
	ULONG			tmp;
	USHORT			CapabilityInfo;
	UCHAR			DA[MAC_ADDR_LEN], SA[MAC_ADDR_LEN];
	USHORT			DLSTimeOut;
	SHORT			i;
	ULONG			Timeout;
	BOOLEAN			TimerCancelled;
	PRT_802_11_DLS	pDLS = NULL;

	if (!PeerDlsReqSanity(pAd, Elem->Msg, Elem->MsgLen, DA, SA, &CapabilityInfo, &DLSTimeOut))
		return;

	DBGPRINT(RT_DEBUG_TRACE,("DLS - PeerDlsReqAction() from %02x:%02x:%02x:%02x:%02x:%02x\n", SA[0], SA[1], SA[2], SA[3], SA[4], SA[5]));

	NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);  //Get an unused nonpaged memory
	if (NStatus != NDIS_STATUS_SUCCESS)
	{
		DBGPRINT(RT_DEBUG_ERROR,("DLS - PeerDlsReqAction() allocate memory failed \n"));
		return;
	}

	if (!INFRA_ON(pAd))
	{
		StatusCode = MLME_REQUEST_DECLINED;
	}
	else if (!pAd->CommonCfg.bWmmCapable)
	{
		StatusCode = MLME_DEST_STA_IS_NOT_A_QSTA;
	}
	else if (!pAd->CommonCfg.bDLSCapable)
	{
		StatusCode = MLME_REQUEST_DECLINED;
	}
	else
	{
		// find table to update parameters
		for (i=MAX_NUM_OF_DLS_ENTRY-1; i>=0; i--)
		{
			if (pAd->StaCfg.DLSEntry[i].Valid && MAC_ADDR_EQUAL(SA, pAd->StaCfg.DLSEntry[i].MacAddr))
			{
				if (pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
					pAd->StaCfg.DLSEntry[i].Status = DLS_WAIT_KEY;
				else
				{
					RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
					pAd->StaCfg.DLSEntry[i].Status = DLS_FINISH;
				}

				pAd->StaCfg.DLSEntry[i].TimeOut = DLSTimeOut;
				pAd->StaCfg.DLSEntry[i].CountDownTimer = DLSTimeOut;
				pDLS = &pAd->StaCfg.DLSEntry[i];
				break;
			}
		}

		// can not find in table, create a new one
		if (i < 0)
		{
			DBGPRINT(RT_DEBUG_TRACE,("DLS - PeerDlsReqAction() can not find same entry \n"));
			for (i=(MAX_NUM_OF_DLS_ENTRY-1); i>=MAX_NUM_OF_INIT_DLS_ENTRY; i--)
			{
				if (!pAd->StaCfg.DLSEntry[i].Valid)
				{
					if (pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
						pAd->StaCfg.DLSEntry[i].Status = DLS_WAIT_KEY;
					else
					{
						RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
						pAd->StaCfg.DLSEntry[i].Status = DLS_FINISH;
					}

					pAd->StaCfg.DLSEntry[i].Valid = TRUE;
					pAd->StaCfg.DLSEntry[i].TimeOut = DLSTimeOut;
					pAd->StaCfg.DLSEntry[i].CountDownTimer = DLSTimeOut;
					NdisMoveMemory(pAd->StaCfg.DLSEntry[i].MacAddr, SA, MAC_ADDR_LEN);
					pDLS = &pAd->StaCfg.DLSEntry[i];
					break;
				}
			}
		}
		StatusCode = MLME_SUCCESS;

		// can not find in table, create a new one
		if (i < 0)
		{
			StatusCode = MLME_QOS_UNSPECIFY;
			DBGPRINT(RT_DEBUG_ERROR,("DLS - PeerDlsReqAction() DLSEntry table full(only can support %d DLS session) \n", MAX_NUM_OF_DLS_ENTRY - MAX_NUM_OF_INIT_DLS_ENTRY));
		}
		else
		{
			DBGPRINT(RT_DEBUG_ERROR,("DLS - PeerDlsReqAction() use entry(%d) %02x:%02x:%02x:%02x:%02x:%02x\n",
				i, SA[0], SA[1], SA[2], SA[3], SA[4], SA[5]));
		}
	}

	MgtMacHeaderInit(pAd, &DlsRspHdr, SUBTYPE_ACTION, 0, pAd->CommonCfg.Bssid, pAd->CommonCfg.Bssid);

	// Build basic frame first
	if (StatusCode == MLME_SUCCESS)
	{
		MakeOutgoingFrame(pOutBuffer,				&FrameLen,
						sizeof(HEADER_802_11),		&DlsRspHdr,
						1,							&Category,
						1,							&Action,
						2,							&StatusCode,
						6,							SA,
						6,							pAd->CurrentAddress,
						2,							&pAd->StaActive.CapabilityInfo,
						1,							&SupRateIe,
						1,							&pAd->MlmeAux.SupRateLen,
						pAd->MlmeAux.SupRateLen,	pAd->MlmeAux.SupRate,
						END_OF_ARGS);

		if (pAd->MlmeAux.ExtRateLen != 0)
		{
			MakeOutgoingFrame(pOutBuffer + FrameLen,	&tmp,
							  1,						&ExtRateIe,
							  1,						&pAd->MlmeAux.ExtRateLen,
							  pAd->MlmeAux.ExtRateLen, 	pAd->MlmeAux.ExtRate,
							  END_OF_ARGS);
			FrameLen += tmp;
		}

		if (pDLS && (pDLS->Status != DLS_FINISH))
		{
			RTMPCancelTimer(&pDLS->Timer, &TimerCancelled);
			Timeout = DLS_TIMEOUT;
			RTMPSetTimer(&pDLS->Timer, Timeout);
		}
	}
	else
	{
		MakeOutgoingFrame(pOutBuffer,				&FrameLen,
						sizeof(HEADER_802_11),		&DlsRspHdr,
						1,							&Category,
						1,							&Action,
						2,							&StatusCode,
						6,							SA,
						6,							pAd->CurrentAddress,
						END_OF_ARGS);
	}

	MiniportMMRequest(pAd, pOutBuffer, FrameLen);
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
VOID PeerDlsRspAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
	USHORT			CapabilityInfo;
	UCHAR			DA[MAC_ADDR_LEN], SA[MAC_ADDR_LEN];
	USHORT			StatusCode;
	SHORT			i;
	BOOLEAN			TimerCancelled;

	if (!pAd->CommonCfg.bDLSCapable)
		return;

	if (!INFRA_ON(pAd))
		return;

	if (!PeerDlsRspSanity(pAd, Elem->Msg, Elem->MsgLen, DA, SA, &CapabilityInfo, &StatusCode))
		return;

	DBGPRINT(RT_DEBUG_TRACE,("DLS - PeerDlsRspAction() from %02x:%02x:%02x:%02x:%02x:%02x with StatusCode=%d, CapabilityInfo=0x%x\n",
		SA[0], SA[1], SA[2], SA[3], SA[4], SA[5], StatusCode, CapabilityInfo));

	for (i=0; i<MAX_NUM_OF_INIT_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && MAC_ADDR_EQUAL(SA, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			if (StatusCode == MLME_SUCCESS)
			{
				if (pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
				{
					// If support WPA or WPA2, start STAKey hand shake,
					// If failed hand shake, just tear down peer DLS
					if (RTMPSendSTAKeyRequest(pAd, pAd->StaCfg.DLSEntry[i].MacAddr) != NDIS_STATUS_SUCCESS)
					{
						MLME_DLS_REQ_STRUCT	MlmeDlsReq;
						USHORT				reason = REASON_QOS_CIPHER_NOT_SUPPORT;

						DlsParmFill(pAd, &MlmeDlsReq, &pAd->StaCfg.DLSEntry[i], reason);
						MlmeEnqueue(pAd, DLS_STATE_MACHINE, MT2_MLME_DLS_TEAR_DOWN, sizeof(MLME_DLS_REQ_STRUCT), &MlmeDlsReq);
						pAd->StaCfg.DLSEntry[i].Status = DLS_NONE;
						pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
						DBGPRINT(RT_DEBUG_ERROR,("DLS - PeerDlsRspAction failed when call RTMPSendSTAKeyRequest \n"));
					}
					else
					{
						pAd->StaCfg.DLSEntry[i].Status = DLS_WAIT_KEY;
						DBGPRINT(RT_DEBUG_TRACE,("DLS - waiting for STAKey handshake procedure\n"));
					}
				}
				else
				{
					RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
					pAd->StaCfg.DLSEntry[i].Status = DLS_FINISH;
					DBGPRINT(RT_DEBUG_TRACE,("DLS - PeerDlsRspAction() from %02x:%02x:%02x:%02x:%02x:%02x Succeed with WEP or no security\n", SA[0], SA[1], SA[2], SA[3], SA[4], SA[5]));
				}
			}
			else
			{
				// DLS setup procedure failed.
				pAd->StaCfg.DLSEntry[i].Status = DLS_NONE;
				pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
				RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
				DBGPRINT(RT_DEBUG_ERROR,("DLS - PeerDlsRspAction failed with StatusCode=%d \n", StatusCode));
			}
		}
	}

	if (i >= MAX_NUM_OF_INIT_DLS_ENTRY)
	{
		DBGPRINT(RT_DEBUG_TRACE,("DLS - PeerDlsRspAction() update timeout value \n"));
		for (i=(MAX_NUM_OF_DLS_ENTRY-1); i>=MAX_NUM_OF_INIT_DLS_ENTRY; i--)
		{
			if (pAd->StaCfg.DLSEntry[i].Valid && MAC_ADDR_EQUAL(SA, pAd->StaCfg.DLSEntry[i].MacAddr))
			{
				if (StatusCode == MLME_SUCCESS)
				{
					if (pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
					{
						// If support WPA or WPA2, start STAKey hand shake,
						// If failed hand shake, just tear down peer DLS
						if (RTMPSendSTAKeyRequest(pAd, pAd->StaCfg.DLSEntry[i].MacAddr) != NDIS_STATUS_SUCCESS)
						{
							MLME_DLS_REQ_STRUCT	MlmeDlsReq;
							USHORT				reason = REASON_QOS_CIPHER_NOT_SUPPORT;

							DlsParmFill(pAd, &MlmeDlsReq, &pAd->StaCfg.DLSEntry[i], reason);
							MlmeEnqueue(pAd, DLS_STATE_MACHINE, MT2_MLME_DLS_TEAR_DOWN, sizeof(MLME_DLS_REQ_STRUCT), &MlmeDlsReq);
							pAd->StaCfg.DLSEntry[i].Status = DLS_NONE;
							pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
							DBGPRINT(RT_DEBUG_ERROR,("DLS - PeerDlsRspAction failed when call RTMPSendSTAKeyRequest \n"));
						}
						else
						{
							pAd->StaCfg.DLSEntry[i].Status = DLS_WAIT_KEY;
							DBGPRINT(RT_DEBUG_TRACE,("DLS - waiting for STAKey handshake procedure\n"));
						}
					}
					else
					{
						RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
						pAd->StaCfg.DLSEntry[i].Status = DLS_FINISH;
						DBGPRINT(RT_DEBUG_TRACE,("DLS - PeerDlsRspAction() from %02x:%02x:%02x:%02x:%02x:%02x Succeed with WEP or no security\n", SA[0], SA[1], SA[2], SA[3], SA[4], SA[5]));
					}
				}
				else
				{
					// DLS setup procedure failed.
					pAd->StaCfg.DLSEntry[i].Status = DLS_NONE;
					pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
					RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
					DBGPRINT(RT_DEBUG_ERROR,("DLS - PeerDlsRspAction failed with StatusCode=%d \n", StatusCode));
				}
			}
		}
	}
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
VOID MlmeDlsTearDownAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
	PUCHAR			pOutBuffer = NULL;
	NDIS_STATUS		NStatus;
	ULONG			FrameLen = 0;
	UCHAR			Category = CATEGORY_DLS;
	UCHAR			Action = ACTION_DLS_TEARDOWN;
	USHORT			ReasonCode = REASON_QOS_UNSPECIFY;
	HEADER_802_11	DlsTearDownHdr;
	PRT_802_11_DLS	pDLS;
	BOOLEAN			TimerCancelled;
	UCHAR			i;

	if(!MlmeDlsReqSanity(pAd, Elem->Msg, Elem->MsgLen, &pDLS, &ReasonCode))
		return;

	DBGPRINT(RT_DEBUG_TRACE,("DLS - MlmeDlsTearDownAction() with ReasonCode=%d \n", ReasonCode));

	NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);  //Get an unused nonpaged memory
	if (NStatus != NDIS_STATUS_SUCCESS)
	{
		DBGPRINT(RT_DEBUG_ERROR,("DLS - MlmeDlsTearDownAction() allocate memory failed \n"));
		return;
	}

	MgtMacHeaderInit(pAd, &DlsTearDownHdr, SUBTYPE_ACTION, 0, pAd->CommonCfg.Bssid, pAd->CommonCfg.Bssid);

	// Build basic frame first
	MakeOutgoingFrame(pOutBuffer,				&FrameLen,
					sizeof(HEADER_802_11),		&DlsTearDownHdr,
					1,							&Category,
					1,							&Action,
					6,							&pDLS->MacAddr,
					6,							pAd->CurrentAddress,
					2,							&ReasonCode,
					END_OF_ARGS);

	MiniportMMRequest(pAd, pOutBuffer, FrameLen);
	RTMPCancelTimer(&pDLS->Timer, &TimerCancelled);

	// Remove key in local dls table entry
	for (i=0; i<MAX_NUM_OF_INIT_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH)
			&& MAC_ADDR_EQUAL(pDLS->MacAddr, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			AsicRemovePairwiseKeyEntry(pAd, i+1);
		}
	}
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
VOID PeerDlsTearDownAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
	UCHAR			DA[MAC_ADDR_LEN], SA[MAC_ADDR_LEN];
	USHORT			ReasonCode;
	UCHAR			i;
	BOOLEAN			TimerCancelled;

	if (!pAd->CommonCfg.bDLSCapable)
		return;

	if (!INFRA_ON(pAd))
		return;

	if (!PeerDlsTearDownSanity(pAd, Elem->Msg, Elem->MsgLen, DA, SA, &ReasonCode))
		return;

	DBGPRINT(RT_DEBUG_TRACE,("DLS - PeerDlsTearDownAction() from %02x:%02x:%02x:%02x:%02x:%02x with ReasonCode=%d\n", SA[0], SA[1], SA[2], SA[3], SA[4], SA[5], ReasonCode));

	// clear local dls table entry
	for (i=0; i<MAX_NUM_OF_INIT_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && MAC_ADDR_EQUAL(SA, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			pAd->StaCfg.DLSEntry[i].Status	= DLS_NONE;
			pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
			RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
			AsicRemovePairwiseKeyEntry(pAd, i+1);
		}
	}

	// clear peer dls table entry
	for (i=MAX_NUM_OF_INIT_DLS_ENTRY; i<MAX_NUM_OF_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && MAC_ADDR_EQUAL(SA, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			pAd->StaCfg.DLSEntry[i].Status	= DLS_NONE;
			pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
			RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
			AsicRemovePairwiseKeyEntry(pAd, i+1);
		}
	}
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
VOID RTMPCheckDLSTimeOut(
	IN PRTMP_ADAPTER	pAd)
{
	ULONG				i;
	MLME_DLS_REQ_STRUCT	MlmeDlsReq;
	USHORT				reason = REASON_QOS_UNSPECIFY;

	if (! pAd->CommonCfg.bDLSCapable)
		return;

	if (! INFRA_ON(pAd))
		return;

	// If timeout value is equaled to zero, it means always not be timeout.

	// update local dls table entry
	for (i=0; i<MAX_NUM_OF_INIT_DLS_ENTRY; i++)
	{
		if ((pAd->StaCfg.DLSEntry[i].Valid) && (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH)
			&& (pAd->StaCfg.DLSEntry[i].TimeOut != 0))
		{
			pAd->StaCfg.DLSEntry[i].CountDownTimer --;

			if (pAd->StaCfg.DLSEntry[i].CountDownTimer == 0)
			{
				reason = REASON_QOS_REQUEST_TIMEOUT;
				pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
				pAd->StaCfg.DLSEntry[i].Status	= DLS_NONE;
				DlsParmFill(pAd, &MlmeDlsReq, &pAd->StaCfg.DLSEntry[i], reason);
				MlmeEnqueue(pAd, DLS_STATE_MACHINE, MT2_MLME_DLS_TEAR_DOWN, sizeof(MLME_DLS_REQ_STRUCT), &MlmeDlsReq);
			}
		}
	}

	// update peer dls table entry
	for (i=MAX_NUM_OF_INIT_DLS_ENTRY; i<MAX_NUM_OF_DLS_ENTRY; i++)
	{
		if ((pAd->StaCfg.DLSEntry[i].Valid) && (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH)
			&& (pAd->StaCfg.DLSEntry[i].TimeOut != 0))
		{
			pAd->StaCfg.DLSEntry[i].CountDownTimer --;

			if (pAd->StaCfg.DLSEntry[i].CountDownTimer == 0)
			{
				reason = REASON_QOS_REQUEST_TIMEOUT;
				pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
				pAd->StaCfg.DLSEntry[i].Status	= DLS_NONE;
				DlsParmFill(pAd, &MlmeDlsReq, &pAd->StaCfg.DLSEntry[i], reason);
				MlmeEnqueue(pAd, DLS_STATE_MACHINE, MT2_MLME_DLS_TEAR_DOWN, sizeof(MLME_DLS_REQ_STRUCT), &MlmeDlsReq);
			}
		}
	}
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
BOOLEAN RTMPRcvFrameDLSCheck(
	IN PRTMP_ADAPTER	pAd,
	IN PHEADER_802_11	pHeader,
	IN ULONG			Len)
{
	ULONG			i;
	BOOLEAN			bSTAKeyFrame = FALSE;
	PEAPOL_PACKET	pEap;
	PUCHAR			pProto, pAddr = NULL;
	PUCHAR			pSTAKey;
	UCHAR			ZeroReplay[LEN_KEY_DESC_REPLAY];
	UCHAR			Mic[16], OldMic[16];
	UCHAR			digest[80];
	UCHAR			DlsPTK[80];
	UCHAR			temp[64];
//	BOOLEAN			TimerCancelled;

	if (! pAd->CommonCfg.bDLSCapable)
		return bSTAKeyFrame;

	if (! INFRA_ON(pAd))
		return bSTAKeyFrame;

	if (! (pHeader->FC.SubType & 0x08))
		return bSTAKeyFrame;

	if (Len < LENGTH_802_11 + 6 + 2 + 2)
		return bSTAKeyFrame;

	pProto	= (PUCHAR)pHeader + LENGTH_802_11 + 2 + 6;	// QOS Control field , 0xAA 0xAA 0xAA 0x00 0x00 0x00
	pAddr	= pHeader->Addr2;

	if (RTMPEqualMemory(EAPOL, pProto, 2) && (pAd->CommonCfg.AuthMode >=  Ndis802_11AuthModeWPA))
	{
		pEap = (PEAPOL_PACKET) (pProto + 2);

		if ((Len >= (LENGTH_802_11 + 6 + 2 + 2 + sizeof(EAPOL_PACKET) - MAX_LEN_OF_RSNIE + 16)) && pEap->KeyDesc.KeyInfo.KeyMic
			&& pEap->KeyDesc.KeyInfo.Install && pEap->KeyDesc.KeyInfo.KeyAck && pEap->KeyDesc.KeyInfo.Secure
			&& pEap->KeyDesc.KeyInfo.EKD_DL && !pEap->KeyDesc.KeyInfo.Error && !pEap->KeyDesc.KeyInfo.Request)
		{
			// First validate replay counter, only accept message with larger replay counter
			// Let equal pass, some AP start with all zero replay counter
			NdisZeroMemory(ZeroReplay, LEN_KEY_DESC_REPLAY);
			if ((RTMPCompareMemory(pEap->KeyDesc.ReplayCounter, pAd->StaCfg.ReplayCounter, LEN_KEY_DESC_REPLAY) != 1) &&
				(RTMPCompareMemory(pEap->KeyDesc.ReplayCounter, ZeroReplay, LEN_KEY_DESC_REPLAY) != 0))
				return bSTAKeyFrame;

			RTMPMoveMemory(pAd->StaCfg.ReplayCounter, pEap->KeyDesc.ReplayCounter, LEN_KEY_DESC_REPLAY);
			DBGPRINT(RT_DEBUG_INFO,("DLS - Sniff replay counter (%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x) Len=%d, KeyDataLen=%d\n",
				pAd->StaCfg.ReplayCounter[0], pAd->StaCfg.ReplayCounter[1], pAd->StaCfg.ReplayCounter[2],
				pAd->StaCfg.ReplayCounter[3], pAd->StaCfg.ReplayCounter[4],	pAd->StaCfg.ReplayCounter[5],
				pAd->StaCfg.ReplayCounter[6], pAd->StaCfg.ReplayCounter[7], Len, pEap->KeyDesc.KeyData[1]));


			// put these code segment to get the replay counter
			if (pAd->StaCfg.PortSecured == WPA_802_1X_PORT_NOT_SECURED)
				return bSTAKeyFrame;

			// Check MIC value
			// Save the MIC and replace with zero
			// use proprietary PTK
			NdisZeroMemory(temp, 64);
			NdisMoveMemory(temp, "IEEE802.11 WIRELESS ACCESS POINT", 32);
			CountPTK(temp, temp, pAd->CommonCfg.Bssid, temp, pAd->CurrentAddress, DlsPTK, LEN_PTK);

			NdisMoveMemory(OldMic, pEap->KeyDesc.KeyMic, LEN_KEY_DESC_MIC);
			NdisZeroMemory(pEap->KeyDesc.KeyMic, LEN_KEY_DESC_MIC);
			if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption3Enabled)
			{
				// AES
				HMAC_SHA1((PUCHAR) pEap, pEap->Len[1] + 4, DlsPTK, LEN_EAP_MICK, digest);
				NdisMoveMemory(Mic,	digest,	LEN_KEY_DESC_MIC);
			}
			else
			{
				hmac_md5(DlsPTK, LEN_EAP_MICK, (PUCHAR) pEap, pEap->Len[1] + 4, Mic);
			}

			if (!NdisEqualMemory(OldMic, Mic, LEN_KEY_DESC_MIC))
			{
				DBGPRINT(RT_DEBUG_ERROR, ("MIC Different in Msg1 of STAKey handshake! \n"));
				return bSTAKeyFrame;
			}
			else
				DBGPRINT(RT_DEBUG_TRACE, ("MIC VALID in Msg1 of STAKey handshake! \n"));
#if 1
			if ((pEap->KeyDesc.KeyData[0] == 0xDD) && (pEap->KeyDesc.KeyData[2] == 0x00) && (pEap->KeyDesc.KeyData[3] == 0x0C)
				&& (pEap->KeyDesc.KeyData[4] == 0x43) && (pEap->KeyDesc.KeyData[5] == 0x02))
			{
				pAddr			= pEap->KeyDesc.KeyData + 8;		// Tpe(1), Len(1), OUI(3), DataType(1), Reserved(2)
				pSTAKey			= pEap->KeyDesc.KeyData + 14;	// Tpe(1), Len(1), OUI(3), DataType(1), Reserved(2), STAKey_Mac_Addr(6)

				DBGPRINT(RT_DEBUG_TRACE,("DLS - Receive STAKey Message-1 from %02x:%02x:%02x:%02x:%02x:%02x Len=%d, KeyDataLen=%d\n",
					pAddr[0], pAddr[1], pAddr[2], pAddr[3], pAddr[4], pAddr[5], Len, pEap->KeyDesc.KeyData[1]));

				bSTAKeyFrame = TRUE;
			}
#else
			if ((pEap->KeyDesc.KeyData[0] == 0xDD) && (pEap->KeyDesc.KeyData[2] == 0x00) && (pEap->KeyDesc.KeyData[3] == 0x0F)
				&& (pEap->KeyDesc.KeyData[4] == 0xAC) && (pEap->KeyDesc.KeyData[5] == 0x02))
			{
				pAddr			= pEap->KeyDesc.KeyData + 8;		// Tpe(1), Len(1), OUI(3), DataType(1), Reserved(2)
				pSTAKey			= pEap->KeyDesc.KeyData + 14;	// Tpe(1), Len(1), OUI(3), DataType(1), Reserved(2), STAKey_Mac_Addr(6)

				DBGPRINT(RT_DEBUG_TRACE,("DLS - Receive STAKey Message-1 from %02x:%02x:%02x:%02x:%02x:%02x Len=%d, KeyDataLen=%d\n",
					pAddr[0], pAddr[1], pAddr[2], pAddr[3], pAddr[4], pAddr[5], Len, pEap->KeyDesc.KeyData[1]));

				bSTAKeyFrame = TRUE;
			}
#endif

		}
		else if (Len >= (LENGTH_802_11 + 6 + 2 + 2 + sizeof(EAPOL_PACKET) - MAX_LEN_OF_RSNIE))
		{
			RTMPMoveMemory(pAd->StaCfg.ReplayCounter, pEap->KeyDesc.ReplayCounter, LEN_KEY_DESC_REPLAY);
			DBGPRINT(RT_DEBUG_INFO,("DLS - Sniff replay counter (%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x) Len=%d, KeyDataLen=%d\n",
				pAd->StaCfg.ReplayCounter[0], pAd->StaCfg.ReplayCounter[1], pAd->StaCfg.ReplayCounter[2],
				pAd->StaCfg.ReplayCounter[3], pAd->StaCfg.ReplayCounter[4],	pAd->StaCfg.ReplayCounter[5],
				pAd->StaCfg.ReplayCounter[6], pAd->StaCfg.ReplayCounter[7], Len, pEap->KeyDesc.KeyData[1]));

		}
	}

	// If timeout value is equaled to zero, it means always not be timeout.
	// update local dls table entry
	for (i=0; i<MAX_NUM_OF_INIT_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && MAC_ADDR_EQUAL(pAddr, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			if (bSTAKeyFrame)
			{
				MlmeEnqueue(pAd, DLS_STATE_MACHINE, MT2_PEER_DLS_STAKEY, Len, pHeader);
#ifdef WIN_NDIS				
				KeSetEvent(&pAd->MLMEEvent, 0, FALSE);
#else
				RTUSBMlmeUp(pAd);
#endif				

				DBGPRINT(RT_DEBUG_TRACE, ("MlmeEnqueue to MT2_PEER_DLS_STAKEY of DLS_STATE_MACHINE \n"));
			}
			else
			{
				// Data frame, update timeout value
				if (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH)
				{
					pAd->StaCfg.DLSEntry[i].CountDownTimer = pAd->StaCfg.DLSEntry[i].TimeOut;
				}
			}
		}
	}

	// update peer dls table entry
	for (i=MAX_NUM_OF_INIT_DLS_ENTRY; i<MAX_NUM_OF_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && MAC_ADDR_EQUAL(pAddr, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			if (bSTAKeyFrame)
			{
				MlmeEnqueue(pAd, DLS_STATE_MACHINE, MT2_PEER_DLS_STAKEY, Len, pHeader);
#ifdef WIN_NDIS				
				KeSetEvent(&pAd->MLMEEvent, 0, FALSE);
#else
				RTUSBMlmeUp(pAd);
#endif								
				DBGPRINT(RT_DEBUG_TRACE, ("MlmeEnqueue to MT2_PEER_DLS_STAKEY of DLS_STATE_MACHINE \n"));
			}
			else
			{
				// Data frame, update timeout value
				if (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH)
				{
					pAd->StaCfg.DLSEntry[i].CountDownTimer = pAd->StaCfg.DLSEntry[i].TimeOut;
				}
			}
		}
	}

	return bSTAKeyFrame;
}

/*
	========================================================================

	Routine	Description:
		Check if the frame can be sent through DLS direct link interface

	Arguments:
		pAd		Pointer	to adapter

	Return Value:
		TRUE		Can be sent through DLS direct link interface
		FALSE		Can not be sent through DLS direct link interface

	Note:

	========================================================================
*/
BOOLEAN	RTMPCheckDLSFrame(
	IN	PRTMP_ADAPTER	pAd,
	IN  PUCHAR          pDA)
{
	BOOLEAN rval = FALSE;
	ULONG	i;

	if (!pAd->CommonCfg.bDLSCapable)
		return FALSE;

	if (!INFRA_ON(pAd))
		return FALSE;

	do{
		// WMM-DLS only works when infrastructure mode
		if (INFRA_ON(pAd))
		{
			// check local dls table entry
			for (i=0; i<MAX_NUM_OF_INIT_DLS_ENTRY; i++)
			{
				if (pAd->StaCfg.DLSEntry[i].Valid && (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH) &&
					MAC_ADDR_EQUAL(pDA, pAd->StaCfg.DLSEntry[i].MacAddr))
				{
					rval = TRUE;
					break;
				}
			}

			// check peer dls table entry
			for (i=MAX_NUM_OF_INIT_DLS_ENTRY; i<MAX_NUM_OF_DLS_ENTRY; i++)
			{
				if (pAd->StaCfg.DLSEntry[i].Valid && (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH) &&
					MAC_ADDR_EQUAL(pDA, pAd->StaCfg.DLSEntry[i].MacAddr))
				{
					rval = TRUE;
					break;
				}
			}
		}
	} while (FALSE);

	return rval;
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
VOID RTMPSendDLSTearDownFrame(
	IN	PRTMP_ADAPTER	pAd,
	IN  PUCHAR          pDA)
{
	NDIS_STATUS		NStatus;
	PUCHAR			pOutBuffer = NULL;
	HEADER_802_11	DlsTearDownHdr;
	ULONG			FrameLen = 0;
	USHORT			Reason = REASON_QOS_QSTA_LEAVING_QBSS;
	UCHAR			Category = CATEGORY_DLS;
	UCHAR			Action = ACTION_DLS_TEARDOWN;
	UCHAR			i;

	// Allocate buffer for transmitting message
	NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
	if (NStatus	!= NDIS_STATUS_SUCCESS)
		return;

	MgtMacHeaderInit(pAd, &DlsTearDownHdr, SUBTYPE_ACTION, 0, pAd->CommonCfg.Bssid, pAd->CommonCfg.Bssid);

	MakeOutgoingFrame(pOutBuffer,				&FrameLen,
					sizeof(HEADER_802_11),		&DlsTearDownHdr,
					1,							&Category,
					1,							&Action,
					6,							pDA,
					6,							pAd->CurrentAddress,
					2,							&Reason,
					END_OF_ARGS);

	MiniportMMRequest(pAd, pOutBuffer, FrameLen);

	// Remove key in local dls table entry
	for (i=0; i<MAX_NUM_OF_INIT_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH)
			&& MAC_ADDR_EQUAL(pDA, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			AsicRemovePairwiseKeyEntry(pAd, i+1);
		}
	}

	// Remove key in peer dls table entry
	for (i=MAX_NUM_OF_INIT_DLS_ENTRY; i<MAX_NUM_OF_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH)
			&& MAC_ADDR_EQUAL(pDA, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			AsicRemovePairwiseKeyEntry(pAd, i+1);
		}
	}

	DBGPRINT(RT_DEBUG_TRACE, ("Send DLS TearDown Frame and remove key in (i=%d) \n", i));
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
NDIS_STATUS RTMPSendSTAKeyRequest(
	IN	PRTMP_ADAPTER	pAd,
	IN	PUCHAR			pDA)
{
	UCHAR				Header802_11[64];
	HEADER_802_11		Header;
	NDIS_STATUS			NStatus;
	ULONG				FrameLen = 0;
	EAPOL_PACKET		Packet;
	UCHAR				Mic[16];
	UCHAR				digest[80];
	PUCHAR				pOutBuffer = NULL;
//	UCHAR				DlsNonce[LEN_KEY_DESC_NONCE];
	UCHAR				temp[64];
	UCHAR				DlsPTK[80];
//	UCHAR				AckRate = RATE_2;
//	USHORT				AckDuration = 0;

	DBGPRINT(RT_DEBUG_TRACE,("DLS - RTMPSendSTAKeyRequest() to %02x:%02x:%02x:%02x:%02x:%02x\n", pDA[0], pDA[1], pDA[2], pDA[3], pDA[4], pDA[5]));

	NdisZeroMemory(Header802_11, 64);
	NdisZeroMemory(&Header, sizeof(HEADER_802_11));

	pAd->Sequence = ((pAd->Sequence) + 1) & (MAX_SEQ_NUMBER);
	MAKE_802_11_HEADER(pAd, Header, pAd->CommonCfg.Bssid, pAd->Sequence, FALSE);
	Header.FC.SubType = SUBTYPE_QDATA;
	Header.FC.Wep = 1;
	NdisMoveMemory(Header802_11, &Header, sizeof(HEADER_802_11));
	NdisMoveMemory(&Header802_11[sizeof(HEADER_802_11)], EAPOL_LLC_SNAP, 8);

	// Zero message body
	NdisZeroMemory(&Packet, sizeof(Packet));
	Packet.Version = EAPOL_VER;
	Packet.Type    = EAPOLKey;
	Packet.Len[1]  = sizeof(KEY_DESCRIPTER) - MAX_LEN_OF_RSNIE + 6 + MAC_ADDR_LEN;		// data field contain KDE andPeer MAC address

	// STAKey Message is as EAPOL-Key(1,1,0,0,G/0,0,0, MIC, 0,Peer MAC KDE)
	if ((pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA) || (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPAPSK))
    {
        Packet.KeyDesc.Type = WPA1_KEY_DESC;
    }
    else if ((pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA2) || (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA2PSK))
    {
        Packet.KeyDesc.Type = WPA2_KEY_DESC;
    }

	// Key descriptor version
	Packet.KeyDesc.KeyInfo.KeyDescVer =
		(((pAd->CommonCfg.PairCipher == Ndis802_11Encryption3Enabled) || (pAd->CommonCfg.GroupCipher == Ndis802_11Encryption3Enabled)) ? (DESC_TYPE_AES) : (DESC_TYPE_TKIP));

	Packet.KeyDesc.KeyInfo.KeyMic	= 1;
	Packet.KeyDesc.KeyInfo.Secure	= 1;
	Packet.KeyDesc.KeyInfo.Request	= 1;

	Packet.KeyDesc.KeyDataLen[1]	= 12;

	// use our own OUI to distinguish proprietary with standard.
	Packet.KeyDesc.KeyData[0]		= 0xDD;
	Packet.KeyDesc.KeyData[1]		= 0x0A;
	Packet.KeyDesc.KeyData[2]		= 0x00;
	Packet.KeyDesc.KeyData[3]		= 0x0C;
	Packet.KeyDesc.KeyData[4]		= 0x43;
	Packet.KeyDesc.KeyData[5]		= 0x03;
	NdisMoveMemory(&Packet.KeyDesc.KeyData[6], pDA, MAC_ADDR_LEN);

	NdisMoveMemory(Packet.KeyDesc.ReplayCounter, pAd->StaCfg.ReplayCounter, LEN_KEY_DESC_REPLAY);

	// Allocate buffer for transmitting message
	NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
	if (NStatus	!= NDIS_STATUS_SUCCESS)
		return NStatus;

	// Prepare EAPOL frame for MIC calculation
	// Be careful, only EAPOL frame is counted for MIC calculation
	MakeOutgoingFrame(pOutBuffer,           &FrameLen,
		              Packet.Len[1] + 4,    &Packet,
		              END_OF_ARGS);

	// use proprietary PTK
	NdisZeroMemory(temp, 64);
	NdisMoveMemory(temp, "IEEE802.11 WIRELESS ACCESS POINT", 32);
	CountPTK(temp, temp, pAd->CommonCfg.Bssid, temp, pAd->CurrentAddress, DlsPTK, LEN_PTK);

	// calculate MIC
	if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption3Enabled)
	{
		// AES
		NdisZeroMemory(digest,	sizeof(digest));
		HMAC_SHA1(pOutBuffer, FrameLen, DlsPTK, LEN_EAP_MICK, digest);
		NdisMoveMemory(Packet.KeyDesc.KeyMic, digest, LEN_KEY_DESC_MIC);
	}
	else
	{
		NdisZeroMemory(Mic,	sizeof(Mic));
		hmac_md5(DlsPTK, LEN_EAP_MICK, pOutBuffer, FrameLen, Mic);
		NdisMoveMemory(Packet.KeyDesc.KeyMic, Mic, LEN_KEY_DESC_MIC);
	}

	// Prepare EAPOL frame for MIC calculation
	// Be careful, only EAPOL frame is counted for MIC calculation
	MakeOutgoingFrame(pOutBuffer,			&FrameLen,
						32,					Header802_11,
						Packet.Len[1] + 4,	&Packet,
						END_OF_ARGS);

	// When TKIP/AES, always use pairwise key table and keyidx=0
	RTMPSendHardEncryptFrame(pAd, pAd->CommonCfg.TxRate, &pAd->SharedKey[BSS0][0], 1, 0, pOutBuffer, FrameLen);
	MlmeFreeMemory(pAd, pOutBuffer);

	DBGPRINT(RT_DEBUG_TRACE, ("RTMPSendSTAKeyRequest- Send STAKey request (NStatus=%x, FrameLen=%d)\n", NStatus, FrameLen));

	return NStatus;
}

/*
    ==========================================================================
    Description:

	IRQL = DISPATCH_LEVEL

    ==========================================================================
 */
NDIS_STATUS RTMPSendSTAKeyHandShake(
	IN	PRTMP_ADAPTER	pAd,
	IN	PUCHAR			pDA)
{
	UCHAR				Header802_11[64];
	HEADER_802_11		Header;
	NDIS_STATUS			NStatus = NDIS_STATUS_SUCCESS;
	ULONG				FrameLen = 0;
	EAPOL_PACKET		Packet;
	UCHAR				Mic[16];
	UCHAR				digest[80];
	PUCHAR				pOutBuffer = NULL;
//	UCHAR				DlsNonce[LEN_KEY_DESC_NONCE];
	UCHAR				temp[64];
	UCHAR				DlsPTK[80];			// Due to dirver can not get PTK, use proprietary PTK
//	UCHAR				AckRate = RATE_2;
//	USHORT				AckDuration = 0;

	DBGPRINT(RT_DEBUG_TRACE,("DLS - RTMPSendSTAKeyHandShake() to %02x:%02x:%02x:%02x:%02x:%02x\n", pDA[0], pDA[1], pDA[2], pDA[3], pDA[4], pDA[5]));

	NdisZeroMemory(Header802_11, 64);
	NdisZeroMemory(&Header, sizeof(HEADER_802_11));

	pAd->Sequence = ((pAd->Sequence) + 1) & (MAX_SEQ_NUMBER);
	MAKE_802_11_HEADER(pAd, Header, pAd->CommonCfg.Bssid, pAd->Sequence, FALSE);
	Header.FC.SubType = SUBTYPE_QDATA;
	Header.FC.Wep = 1;
	NdisMoveMemory(Header802_11, &Header, sizeof(HEADER_802_11));
	NdisMoveMemory(&Header802_11[24], EAPOL_LLC_SNAP, 8);

	// Zero message body
	NdisZeroMemory(&Packet, sizeof(Packet));
	Packet.Version = EAPOL_VER;
	Packet.Type    = EAPOLKey;
	Packet.Len[1]  = sizeof(KEY_DESCRIPTER) - MAX_LEN_OF_RSNIE + 6 + MAC_ADDR_LEN;		// data field contain KDE and Peer MAC address

	// STAKey Message is as EAPOL-Key(1,1,0,0,G/0,0,0, MIC, 0,Peer MAC KDE)
	if ((pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA) || (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPAPSK))
    {
        Packet.KeyDesc.Type = WPA1_KEY_DESC;
    }
    else if ((pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA2) || (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA2PSK))
    {
        Packet.KeyDesc.Type = WPA2_KEY_DESC;
    }

	// Key descriptor version
	Packet.KeyDesc.KeyInfo.KeyDescVer =
		(((pAd->CommonCfg.PairCipher == Ndis802_11Encryption3Enabled) || (pAd->CommonCfg.GroupCipher == Ndis802_11Encryption3Enabled)) ? (DESC_TYPE_AES) : (DESC_TYPE_TKIP));

	Packet.KeyDesc.KeyInfo.KeyMic	= 1;
	Packet.KeyDesc.KeyInfo.Secure	= 1;

	Packet.KeyDesc.KeyDataLen[1]	= 12;

	// use our own OUI to distinguish proprietary with standard.
	Packet.KeyDesc.KeyData[0]		= 0xDD;
	Packet.KeyDesc.KeyData[1]		= 0x0A;
	Packet.KeyDesc.KeyData[2]		= 0x00;
	Packet.KeyDesc.KeyData[3]		= 0x0C;
	Packet.KeyDesc.KeyData[4]		= 0x43;
	Packet.KeyDesc.KeyData[5]		= 0x03;
	NdisMoveMemory(&Packet.KeyDesc.KeyData[6], pDA, MAC_ADDR_LEN);

	NdisMoveMemory(Packet.KeyDesc.ReplayCounter, pAd->StaCfg.ReplayCounter, LEN_KEY_DESC_REPLAY);

	// Allocate buffer for transmitting message
	NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
	if (NStatus	!= NDIS_STATUS_SUCCESS)
		return NStatus;

	// Prepare EAPOL frame for MIC calculation
	// Be careful, only EAPOL frame is counted for MIC calculation
	MakeOutgoingFrame(pOutBuffer,           &FrameLen,
		              Packet.Len[1] + 4,    &Packet,
		              END_OF_ARGS);

	// use proprietary PTK
	NdisZeroMemory(temp, 64);
	NdisMoveMemory(temp, "IEEE802.11 WIRELESS ACCESS POINT", 32);
	CountPTK(temp, temp, pAd->CommonCfg.Bssid, temp, pAd->CurrentAddress, DlsPTK, LEN_PTK);

	// calculate MIC
	if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption3Enabled)
	{
		// AES
		NdisZeroMemory(digest,	sizeof(digest));
		HMAC_SHA1(pOutBuffer, FrameLen, DlsPTK, LEN_EAP_MICK, digest);
		NdisMoveMemory(Packet.KeyDesc.KeyMic, digest, LEN_KEY_DESC_MIC);
	}
	else
	{
		NdisZeroMemory(Mic,	sizeof(Mic));
		hmac_md5(DlsPTK, LEN_EAP_MICK, pOutBuffer, FrameLen, Mic);
		NdisMoveMemory(Packet.KeyDesc.KeyMic, Mic, LEN_KEY_DESC_MIC);
	}

	MakeOutgoingFrame(pOutBuffer,           &FrameLen,
	                  32,					Header802_11,
		              Packet.Len[1] + 4,	&Packet,
		              END_OF_ARGS);

	// When TKIP/AES, always use pairwise key table and keyidx=0
	RTMPSendHardEncryptFrame(pAd, pAd->CommonCfg.TxRate, &pAd->SharedKey[BSS0][0], 1, 0, pOutBuffer, FrameLen);
	MlmeFreeMemory(pAd, pOutBuffer);

	DBGPRINT(RT_DEBUG_TRACE, ("RTMPSendSTAKeyHandShake- Send STAKey Message-2 (NStatus=%x, FrameLen=%d)\n", NStatus, FrameLen));

	return NStatus;
}

VOID DlsTimeout(
	IN PVOID SystemSpecific1,
	IN PVOID FunctionContext,
	IN PVOID SystemSpecific2,
	IN PVOID SystemSpecific3)
{
	MLME_DLS_REQ_STRUCT		MlmeDlsReq;
	USHORT					reason;
	PRT_802_11_DLS			pDLS = (PRT_802_11_DLS)FunctionContext;
	PRTMP_ADAPTER			pAd = pDLS->pAd;

	DBGPRINT(RT_DEBUG_TRACE, ("DlsTimeout - Tear down DLS links (%02x:%02x:%02x:%02x:%02x:%02x)\n",
		pDLS->MacAddr[0], pDLS->MacAddr[1], pDLS->MacAddr[2], pDLS->MacAddr[3], pDLS->MacAddr[4], pDLS->MacAddr[5]));

	if ((pDLS) && (pDLS->Valid))
	{
		reason			= REASON_QOS_REQUEST_TIMEOUT;
		pDLS->Valid		= FALSE;
		pDLS->Status	= DLS_NONE;
		DlsParmFill(pAd, &MlmeDlsReq, pDLS, reason);
		MlmeEnqueue(pAd, DLS_STATE_MACHINE, MT2_MLME_DLS_TEAR_DOWN, sizeof(MLME_DLS_REQ_STRUCT), &MlmeDlsReq);
		MlmeHandler(pAd);
	}
}

VOID PeerDlsStaKeyAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
	ULONG			i;
	BOOLEAN			bSTAKeyFrame = FALSE;
	PEAPOL_PACKET	pEap;
	PUCHAR			pProto, pAddr = NULL;
	PUCHAR			pSTAKey = NULL;
//	UCHAR			ZeroReplay[LEN_KEY_DESC_REPLAY];
//	UCHAR			Mic[16], OldMic[16];
//	UCHAR			digest[80];
//	UCHAR			DlsPTK[80];
//	UCHAR			temp[64];
	BOOLEAN			TimerCancelled;
	PHEADER_802_11	pHeader;
	ULONG			Len;

	DBGPRINT(RT_DEBUG_TRACE,("DLS - PeerDlsStaKeyAction() \n"));

	pHeader = (PHEADER_802_11) Elem->Msg;
	Len = Elem->MsgLen;

	pProto	= (PUCHAR)pHeader + LENGTH_802_11 + 2 + 6;	// QOS Control field , 0xAA 0xAA 0xAA 0x00 0x00 0x00
	pAddr	= pHeader->Addr2;

	if (RTMPEqualMemory(EAPOL, pProto, 2) && (pAd->CommonCfg.AuthMode >=  Ndis802_11AuthModeWPA))
	{
		pEap = (PEAPOL_PACKET) (pProto + 2);

		if ((Len >= (LENGTH_802_11 + 6 + 2 + 2 + sizeof(EAPOL_PACKET) - MAX_LEN_OF_RSNIE + 16)) && pEap->KeyDesc.KeyInfo.KeyMic
			&& pEap->KeyDesc.KeyInfo.Install && pEap->KeyDesc.KeyInfo.KeyAck && pEap->KeyDesc.KeyInfo.Secure
			&& pEap->KeyDesc.KeyInfo.EKD_DL && !pEap->KeyDesc.KeyInfo.Error && !pEap->KeyDesc.KeyInfo.Request)
		{
#if 1
			if ((pEap->KeyDesc.KeyData[0] == 0xDD) && (pEap->KeyDesc.KeyData[2] == 0x00) && (pEap->KeyDesc.KeyData[3] == 0x0C)
				&& (pEap->KeyDesc.KeyData[4] == 0x43) && (pEap->KeyDesc.KeyData[5] == 0x02))
			{
				pAddr			= pEap->KeyDesc.KeyData + 8;		// Tpe(1), Len(1), OUI(3), DataType(1), Reserved(2)
				pSTAKey			= pEap->KeyDesc.KeyData + 14;	// Tpe(1), Len(1), OUI(3), DataType(1), Reserved(2), STAKey_Mac_Addr(6)

				DBGPRINT(RT_DEBUG_TRACE,("DLS - Receive STAKey Message-1 from %02x:%02x:%02x:%02x:%02x:%02x Len=%d, KeyDataLen=%d\n",
					pAddr[0], pAddr[1], pAddr[2], pAddr[3], pAddr[4], pAddr[5], Len, pEap->KeyDesc.KeyData[1]));

				bSTAKeyFrame = TRUE;
			}
#else
			if ((pEap->KeyDesc.KeyData[0] == 0xDD) && (pEap->KeyDesc.KeyData[2] == 0x00) && (pEap->KeyDesc.KeyData[3] == 0x0F)
				&& (pEap->KeyDesc.KeyData[4] == 0xAC) && (pEap->KeyDesc.KeyData[5] == 0x02))
			{
				pAddr			= pEap->KeyDesc.KeyData + 8;		// Tpe(1), Len(1), OUI(3), DataType(1), Reserved(2)
				pSTAKey			= pEap->KeyDesc.KeyData + 14;	// Tpe(1), Len(1), OUI(3), DataType(1), Reserved(2), STAKey_Mac_Addr(6)

				DBGPRINT(RT_DEBUG_TRACE,("DLS - Receive STAKey Message-1 from %02x:%02x:%02x:%02x:%02x:%02x Len=%d, KeyDataLen=%d\n",
					pAddr[0], pAddr[1], pAddr[2], pAddr[3], pAddr[4], pAddr[5], Len, pEap->KeyDesc.KeyData[1]));

				bSTAKeyFrame = TRUE;
			}
#endif

		}
		else if (Len >= (LENGTH_802_11 + 6 + 2 + 2 + sizeof(EAPOL_PACKET) - MAX_LEN_OF_RSNIE))
		{
		}
	}

    if (pSTAKey == NULL) return;

	// If timeout value is equaled to zero, it means always not be timeout.
	// update local dls table entry
	for (i=0; i<MAX_NUM_OF_INIT_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && MAC_ADDR_EQUAL(pAddr, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			if (bSTAKeyFrame)
			{
				// STAKey frame, add pairwise key table
				pAd->StaCfg.DLSEntry[i].Status = DLS_FINISH;
				RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
				AsicAddPairwiseKeyEntry(pAd,
										pAddr,
										(UCHAR)(i + 1),		// pairwise key entry 0 reserved for AP
										pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg,
										&pSTAKey[0],
										&pSTAKey[16],
										&pSTAKey[24]);

				DBGPRINT(RT_DEBUG_TRACE,("DLS - Receive STAKey Message-1 (Peer STA MAC Address STAKey) \n"));

				RTMPSendSTAKeyHandShake(pAd, pAd->StaCfg.DLSEntry[i].MacAddr);

				DBGPRINT(RT_DEBUG_TRACE,("DLS - Finish STAKey handshake procedure (Initiator side)\n"));
			}
			else
			{
				// Data frame, update timeout value
				if (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH)
				{
					pAd->StaCfg.DLSEntry[i].CountDownTimer = pAd->StaCfg.DLSEntry[i].TimeOut;
				}
			}
		}
	}

	// update peer dls table entry
	for (i=MAX_NUM_OF_INIT_DLS_ENTRY; i<MAX_NUM_OF_DLS_ENTRY; i++)
	{
		if (pAd->StaCfg.DLSEntry[i].Valid && MAC_ADDR_EQUAL(pAddr, pAd->StaCfg.DLSEntry[i].MacAddr))
		{
			if (bSTAKeyFrame)
			{
				// STAKey frame, add pairwise key table, and send STAkey Msg-2
				pAd->StaCfg.DLSEntry[i].Status = DLS_FINISH;
				RTMPCancelTimer(&pAd->StaCfg.DLSEntry[i].Timer, &TimerCancelled);
				AsicAddPairwiseKeyEntry(pAd,
										pAddr,
										(UCHAR)(i + 1),		// pairwise key entry 0 reserved for AP
										pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg,
										&pSTAKey[0],
										&pSTAKey[16],
										&pSTAKey[24]);

				DBGPRINT(RT_DEBUG_TRACE,("DLS - Receive STAKey Message-1 (Initiator STA MAC Address STAKey)\n"));

				// If support WPA or WPA2, start STAKey hand shake,
				// If failed hand shake, just tear down peer DLS
				if (RTMPSendSTAKeyHandShake(pAd, pAddr) != NDIS_STATUS_SUCCESS)
				{
					MLME_DLS_REQ_STRUCT	MlmeDlsReq;
					USHORT				reason = REASON_QOS_CIPHER_NOT_SUPPORT;

					pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
					pAd->StaCfg.DLSEntry[i].Status	= DLS_NONE;
					DlsParmFill(pAd, &MlmeDlsReq, &pAd->StaCfg.DLSEntry[i], reason);
					MlmeEnqueue(pAd, DLS_STATE_MACHINE, MT2_MLME_DLS_TEAR_DOWN, sizeof(MLME_DLS_REQ_STRUCT), &MlmeDlsReq);
				}
				else
				{
					DBGPRINT(RT_DEBUG_TRACE,("DLS - Finish STAKey handshake procedure (Peer side)\n"));
				}
			}
			else
			{
				// Data frame, update timeout value
				if (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH)
				{
					pAd->StaCfg.DLSEntry[i].CountDownTimer = pAd->StaCfg.DLSEntry[i].TimeOut;
				}
			}
		}
	}
}

